/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.store.local;

import ghidra.framework.store.CheckoutType;
import ghidra.framework.store.ExclusiveCheckoutException;
import ghidra.framework.store.ItemCheckoutStatus;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.framework.store.local.LocalFolderItem;
import ghidra.util.datastruct.LongObjectHashtable;
import ghidra.util.xml.GenericXMLOutputter;
import ghidra.util.xml.XmlUtilities;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InvalidObjectException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.jdom.Content;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

class CheckoutManager {
    static final String CHECKOUTS_FILE = "checkout.dat";
    private LocalFolderItem item;
    private long nextCheckoutId = 1L;
    private LongObjectHashtable<ItemCheckoutStatus> checkouts;

    CheckoutManager(LocalFolderItem item, boolean create) throws IOException {
        this.item = item;
        if (create) {
            this.checkouts = new LongObjectHashtable();
            this.writeCheckoutsFile();
        }
    }

    private File getCheckoutsFile() {
        return new File(this.item.getDataDir(), CHECKOUTS_FILE);
    }

    synchronized ItemCheckoutStatus newCheckout(CheckoutType checkoutType, String user, int version, String projectPath) throws IOException {
        this.validate();
        if (checkoutType == null) {
            throw new IllegalArgumentException("checkoutType must be specified");
        }
        ItemCheckoutStatus[] coList = this.getAllCheckouts();
        if (coList.length != 0) {
            if (checkoutType != CheckoutType.NORMAL) {
                return null;
            }
            if (coList[0].getCheckoutType() == CheckoutType.TRANSIENT) {
                throw new ExclusiveCheckoutException("File temporarily checked out exclusively by: " + coList[0].getUser());
            }
            if (coList[0].getCheckoutType() == CheckoutType.EXCLUSIVE) {
                throw new ExclusiveCheckoutException("File checked out exclusively to another project by: " + coList[0].getUser());
            }
        }
        ItemCheckoutStatus coStatus = new ItemCheckoutStatus(this.nextCheckoutId++, checkoutType, user, version, new Date().getTime(), projectPath);
        this.checkouts.put(coStatus.getCheckoutId(), (Object)coStatus);
        if (checkoutType != CheckoutType.TRANSIENT) {
            this.writeCheckoutsFile();
        }
        this.item.log("checkout (" + coStatus.getCheckoutId() + ") granted", user);
        return coStatus;
    }

    synchronized void updateCheckout(long checkoutId, int version) throws IOException {
        this.validate();
        ItemCheckoutStatus coStatus = (ItemCheckoutStatus)this.checkouts.remove(checkoutId);
        if (coStatus != null) {
            CheckoutType checkoutType = coStatus.getCheckoutType();
            coStatus = new ItemCheckoutStatus(checkoutId, checkoutType, coStatus.getUser(), version, new Date().getTime(), coStatus.getProjectPath());
            this.checkouts.put(checkoutId, (Object)coStatus);
            if (checkoutType != CheckoutType.TRANSIENT) {
                try {
                    this.writeCheckoutsFile();
                }
                catch (IOException e) {
                    this.item.log("ERROR! failed to update checkout version", coStatus.getUser());
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void endCheckout(long checkoutId) throws IOException {
        this.validate();
        ItemCheckoutStatus coStatus = (ItemCheckoutStatus)this.checkouts.remove(checkoutId);
        if (coStatus != null) {
            this.item.log("checkout (" + checkoutId + ") ended", coStatus.getUser());
            if (coStatus.getCheckoutType() != CheckoutType.TRANSIENT) {
                boolean success = false;
                try {
                    this.writeCheckoutsFile();
                    success = true;
                }
                finally {
                    if (!success) {
                        this.checkouts.put(checkoutId, (Object)coStatus);
                    }
                }
            }
        }
    }

    synchronized boolean isCheckedOut(int version) throws IOException {
        long[] ids;
        this.validate();
        for (long id : ids = this.checkouts.getKeys()) {
            ItemCheckoutStatus coStatus = (ItemCheckoutStatus)this.checkouts.get(id);
            if (coStatus.getCheckoutVersion() != version) continue;
            return true;
        }
        return false;
    }

    synchronized boolean isCheckedOut() throws IOException {
        this.validate();
        return this.checkouts.size() != 0;
    }

    synchronized ItemCheckoutStatus getCheckout(long checkoutId) throws IOException {
        this.validate();
        return (ItemCheckoutStatus)this.checkouts.get(checkoutId);
    }

    synchronized ItemCheckoutStatus[] getAllCheckouts() throws IOException {
        this.validate();
        long[] ids = this.checkouts.getKeys();
        Arrays.sort(ids);
        ItemCheckoutStatus[] list = new ItemCheckoutStatus[ids.length];
        for (int i = 0; i < ids.length; ++i) {
            list[i] = (ItemCheckoutStatus)this.checkouts.get(ids[i]);
        }
        return list;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validate() throws IOException {
        if (LocalFileSystem.isRefreshRequired()) {
            this.checkouts = null;
        }
        if (this.checkouts == null) {
            LongObjectHashtable<ItemCheckoutStatus> oldCheckouts = this.checkouts;
            long oldNextCheckoutId = this.nextCheckoutId;
            boolean success = false;
            try {
                this.readCheckoutsFile();
                success = true;
            }
            finally {
                if (!success) {
                    this.nextCheckoutId = oldNextCheckoutId;
                    this.checkouts = oldCheckouts;
                }
            }
        }
    }

    private void readCheckoutsFile() throws IOException {
        this.checkouts = new LongObjectHashtable();
        File checkoutsFile = this.getCheckoutsFile();
        if (!checkoutsFile.exists()) {
            return;
        }
        FileInputStream istream = new FileInputStream(checkoutsFile);
        BufferedInputStream bis = new BufferedInputStream(istream);
        try {
            SAXBuilder sax = XmlUtilities.createSecureSAXBuilder((boolean)false, (boolean)false);
            Document doc = sax.build((InputStream)bis);
            Element root = doc.getRootElement();
            String nextId = root.getAttributeValue("NEXT_ID");
            try {
                this.nextCheckoutId = Long.parseLong(nextId);
            }
            catch (NumberFormatException e) {
                throw new IOException("Invalid checkouts file: " + checkoutsFile);
            }
            List elementList = root.getChildren("CHECKOUT");
            Iterator iter = elementList.iterator();
            while (iter.hasNext()) {
                ItemCheckoutStatus coStatus = this.parseCheckoutElement((Element)iter.next());
                this.checkouts.put(coStatus.getCheckoutId(), (Object)coStatus);
            }
        }
        catch (JDOMException je) {
            throw new InvalidObjectException("Invalid checkouts file: " + checkoutsFile);
        }
        finally {
            istream.close();
        }
    }

    ItemCheckoutStatus parseCheckoutElement(Element coElement) throws JDOMException {
        try {
            long checkoutId = Long.parseLong(coElement.getAttributeValue("ID"));
            String user = coElement.getAttributeValue("USER");
            int checkoutVersion = Integer.parseInt(coElement.getAttributeValue("VERSION"));
            long time = Long.parseLong(coElement.getAttributeValue("TIME"));
            String projectPath = coElement.getAttributeValue("PROJECT");
            String val = coElement.getAttributeValue("EXCLUSIVE");
            boolean exclusive = val != null ? Boolean.valueOf(val) : false;
            CheckoutType checkoutType = exclusive ? CheckoutType.EXCLUSIVE : CheckoutType.NORMAL;
            return new ItemCheckoutStatus(checkoutId, checkoutType, user, checkoutVersion, time, projectPath);
        }
        catch (NumberFormatException e) {
            throw new JDOMException("Bad CHECKOUT element");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeCheckoutsFile() throws IOException {
        long[] ids;
        Element root = new Element("CHECKOUT_LIST");
        root.setAttribute("NEXT_ID", Long.toString(this.nextCheckoutId));
        for (long id : ids = this.checkouts.getKeys()) {
            ItemCheckoutStatus coStatus = (ItemCheckoutStatus)this.checkouts.get(id);
            if (coStatus.getCheckoutType() == CheckoutType.TRANSIENT) continue;
            root.addContent((Content)this.getCheckoutElement(coStatus));
        }
        File checkoutsFile = this.getCheckoutsFile();
        File tmpFile = new File(checkoutsFile.getParentFile(), checkoutsFile.getName() + ".new");
        tmpFile.delete();
        FileOutputStream ostream = new FileOutputStream(tmpFile);
        try (BufferedOutputStream bos = new BufferedOutputStream(ostream);){
            Document doc = new Document(root);
            GenericXMLOutputter xmlout = new GenericXMLOutputter();
            xmlout.output(doc, (OutputStream)bos);
        }
        File oldFile = null;
        if (checkoutsFile.exists()) {
            oldFile = new File(checkoutsFile.getParentFile(), checkoutsFile.getName() + ".bak");
            oldFile.delete();
            if (!checkoutsFile.renameTo(oldFile)) {
                throw new IOException("Failed to update checkouts: " + this.item.getPathName());
            }
        }
        if (!tmpFile.renameTo(checkoutsFile)) {
            if (oldFile != null) {
                oldFile.renameTo(checkoutsFile);
            }
            throw new IOException("Failed to update checkouts: " + this.item.getPathName());
        }
        if (oldFile != null) {
            oldFile.delete();
        }
    }

    Element getCheckoutElement(ItemCheckoutStatus coStatus) {
        Element element = new Element("CHECKOUT");
        element.setAttribute("ID", Long.toString(coStatus.getCheckoutId()));
        element.setAttribute("USER", coStatus.getUser());
        element.setAttribute("VERSION", Integer.toString(coStatus.getCheckoutVersion()));
        element.setAttribute("TIME", Long.toString(coStatus.getCheckoutTime()));
        String projectPath = coStatus.getProjectPath();
        if (projectPath != null) {
            element.setAttribute("PROJECT", projectPath);
        }
        element.setAttribute("EXCLUSIVE", Boolean.toString(coStatus.getCheckoutType() == CheckoutType.EXCLUSIVE));
        return element;
    }
}

