/*
 * Decompiled with CFR 0.152.
 */
package jadx.core.utils.files;

import com.android.dex.Dex;
import jadx.core.utils.AsmUtils;
import jadx.core.utils.exceptions.DecodeException;
import jadx.core.utils.exceptions.JadxException;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.utils.files.DexFile;
import jadx.core.utils.files.FileUtils;
import jadx.core.utils.files.JavaToDex;
import jadx.core.utils.files.ZipSecurity;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.jetbrains.annotations.Nullable;
import org.jf.smali.Smali;
import org.jf.smali.SmaliOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InputFile {
    private static final Logger LOG = LoggerFactory.getLogger(InputFile.class);
    private final File file;
    private final List<DexFile> dexFiles = new ArrayList<DexFile>();

    public static void addFilesFrom(File file, List<InputFile> list, boolean skipSources) throws IOException, DecodeException {
        InputFile inputFile = new InputFile(file);
        inputFile.searchDexFiles(skipSources);
        list.add(inputFile);
    }

    private InputFile(File file) throws IOException {
        if (!file.exists()) {
            throw new IOException("File not found: " + file.getAbsolutePath());
        }
        this.file = file;
    }

    private void searchDexFiles(boolean skipSources) throws IOException, DecodeException {
        String fileName = this.file.getName();
        if (fileName.endsWith(".dex")) {
            this.addDexFile(fileName, new Dex(this.file), this.file.toPath());
            return;
        }
        if (fileName.endsWith(".smali")) {
            Path output = FileUtils.createTempFile(".dex");
            SmaliOptions options = new SmaliOptions();
            options.outputDexFile = output.toAbsolutePath().toString();
            Smali.assemble((SmaliOptions)options, (String[])new String[]{this.file.getAbsolutePath()});
            this.addDexFile("", new Dex(output.toFile()), output);
            return;
        }
        if (fileName.endsWith(".class")) {
            for (Path path : InputFile.loadFromClassFile(this.file)) {
                this.addDexFile(path);
            }
            return;
        }
        if (FileUtils.isApkFile(this.file) || FileUtils.isZipDexFile(this.file)) {
            this.loadFromZip(".dex");
            return;
        }
        if (fileName.endsWith(".jar") || fileName.endsWith(".aar")) {
            if (this.loadFromZip(".dex")) {
                return;
            }
            if (fileName.endsWith(".jar")) {
                for (Path path : InputFile.loadFromJar(this.file.toPath())) {
                    this.addDexFile(path);
                }
                return;
            }
            if (fileName.endsWith(".aar")) {
                this.loadFromZip(".jar");
                return;
            }
            return;
        }
        if (skipSources) {
            return;
        }
        LOG.warn("No dex files found in {}", (Object)this.file);
    }

    private void addDexFile(Path path) throws IOException {
        this.addDexFile(path.getFileName().toString(), path);
    }

    private void addDexFile(String fileName, Path path) throws IOException {
        this.addDexFile(fileName, new Dex(Files.readAllBytes(path)), path);
    }

    private void addDexFile(String fileName, Dex dexBuf, Path path) {
        this.dexFiles.add(new DexFile(this, fileName, dexBuf, path));
    }

    private boolean loadFromZip(String ext) throws IOException, DecodeException {
        int index = 0;
        try (ZipFile zf = new ZipFile(this.file);){
            String instantRunDexSuffix = "classes" + ext;
            Enumeration<? extends ZipEntry> e = zf.entries();
            block26: while (e.hasMoreElements()) {
                ZipEntry entry = e.nextElement();
                if (!ZipSecurity.isValidZipEntry(entry)) continue;
                String entryName = entry.getName();
                InputStream inputStream = zf.getInputStream(entry);
                Throwable throwable = null;
                try {
                    if (entryName.startsWith("classes") && entryName.endsWith(ext) || entryName.endsWith(instantRunDexSuffix)) {
                        switch (ext) {
                            case ".dex": {
                                Path path = this.makeDexBuf(entryName, inputStream);
                                if (path == null) continue block26;
                                this.addDexFile(entryName, path);
                                ++index;
                                break;
                            }
                            case ".jar": {
                                ++index;
                                Path jarFile = FileUtils.createTempFile(entryName);
                                Files.copy(inputStream, jarFile, StandardCopyOption.REPLACE_EXISTING);
                                for (Path p : InputFile.loadFromJar(jarFile)) {
                                    this.addDexFile(entryName, p);
                                }
                                continue block26;
                            }
                            default: {
                                throw new JadxRuntimeException("Unexpected extension in zip: " + ext);
                            }
                        }
                        continue;
                    }
                    if (!entryName.equals("instant-run.zip") || !ext.equals(".dex")) continue;
                    Path jarFile = FileUtils.createTempFile("instant-run.zip");
                    Files.copy(inputStream, jarFile, StandardCopyOption.REPLACE_EXISTING);
                    InputFile tempFile = new InputFile(jarFile.toFile());
                    tempFile.loadFromZip(ext);
                    List<DexFile> dexFiles = tempFile.getDexFiles();
                    if (dexFiles.isEmpty()) continue;
                    index += dexFiles.size();
                    this.dexFiles.addAll(dexFiles);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (inputStream == null) continue;
                    if (throwable != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    inputStream.close();
                }
            }
        }
        return index > 0;
    }

    @Nullable
    private Path makeDexBuf(String entryName, InputStream inputStream) {
        try {
            Path path = FileUtils.createTempFile(".dex");
            Files.copy(inputStream, path, StandardCopyOption.REPLACE_EXISTING);
            return path;
        }
        catch (Exception e) {
            LOG.error("Failed to load file: {}, error: {}", new Object[]{entryName, e.getMessage(), e});
            return null;
        }
    }

    private static List<Path> loadFromJar(Path jar) throws DecodeException {
        JavaToDex j2d = new JavaToDex();
        try {
            LOG.info("converting to dex: {} ...", (Object)jar.getFileName());
            List<Path> pathList = j2d.convert(jar);
            if (pathList.isEmpty()) {
                throw new JadxException("Empty dx output");
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("result dex files: {}", pathList);
            }
            List<Path> list = pathList;
            return list;
        }
        catch (Exception e) {
            throw new DecodeException("java class to dex conversion error:\n " + e.getMessage(), e);
        }
        finally {
            if (j2d.isError()) {
                LOG.warn("dx message: {}", (Object)j2d.getDxErrors());
            }
        }
    }

    private static List<Path> loadFromClassFile(File file) throws IOException, DecodeException {
        Path outFile = FileUtils.createTempFile(".jar");
        try (JarOutputStream jo = new JarOutputStream(Files.newOutputStream(outFile, new OpenOption[0]));){
            String clsName = AsmUtils.getNameFromClassFile(file);
            if (clsName == null || !ZipSecurity.isValidZipEntryName(clsName)) {
                throw new IOException("Can't read class name from file: " + file);
            }
            FileUtils.addFileToJar(jo, file, clsName + ".class");
        }
        return InputFile.loadFromJar(outFile);
    }

    public File getFile() {
        return this.file;
    }

    public List<DexFile> getDexFiles() {
        return this.dexFiles;
    }

    public String toString() {
        return this.file.getAbsolutePath();
    }
}

