/*
 * Decompiled with CFR 0.152.
 */
package docking.widgets.textfield;

import docking.DockingUtils;
import docking.ToolTipManager;
import docking.util.GraphicsUtils;
import ghidra.util.SystemUtilities;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.AbstractDocument;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.DocumentFilter;

public class IntegerTextField {
    private JTextField textField;
    private boolean isHexMode = false;
    private boolean allowsNegative = true;
    private boolean allowsHexPrefix = true;
    private boolean showNumbericDecoration = true;
    private BigInteger maxValue;
    private List<ChangeListener> listeners = new ArrayList<ChangeListener>();

    public IntegerTextField() {
        this(5, null);
    }

    public IntegerTextField(int columns) {
        this(columns, null);
    }

    public IntegerTextField(int columns, long initialValue) {
        this(columns, BigInteger.valueOf(initialValue));
    }

    public IntegerTextField(int columns, BigInteger initialValue) {
        this.textField = new MyTextField(columns);
        this.setValue(initialValue);
        this.textField.getDocument().addDocumentListener(new DocumentListener(){

            @Override
            public void removeUpdate(DocumentEvent e) {
                SystemUtilities.runSwingLater(() -> IntegerTextField.this.valueChanged());
            }

            @Override
            public void insertUpdate(DocumentEvent e) {
                SystemUtilities.runSwingLater(() -> IntegerTextField.this.valueChanged());
            }

            @Override
            public void changedUpdate(DocumentEvent e) {
                SystemUtilities.runSwingLater(() -> IntegerTextField.this.valueChanged());
            }
        });
    }

    public void addChangeListener(ChangeListener listener) {
        this.listeners.add(listener);
    }

    public void removeChangeListener(ChangeListener listener) {
        this.listeners.remove(listener);
    }

    public BigInteger getValue() {
        String text = this.textField.getText();
        return this.computeValueFromString(text);
    }

    public int getIntValue() {
        BigInteger currentValue = this.getValue();
        if (currentValue == null) {
            return 0;
        }
        return currentValue.intValue();
    }

    public long getLongValue() {
        BigInteger currentValue = this.getValue();
        if (currentValue == null) {
            return 0L;
        }
        return currentValue.longValue();
    }

    public void setValue(long newValue) {
        this.setValue(BigInteger.valueOf(newValue));
    }

    public void setValue(int newValue) {
        this.setValue(BigInteger.valueOf(newValue));
    }

    public void setValue(BigInteger newValue) {
        if (!this.allowsNegative && newValue != null && newValue.compareTo(BigInteger.ZERO) < 0) {
            newValue = null;
        }
        this.updateTextField(newValue);
    }

    public void setShowNumberMode(boolean show) {
        this.showNumbericDecoration = show;
        this.textField.repaint();
    }

    public void setHexMode() {
        BigInteger currentValue = this.getValue();
        this.isHexMode = true;
        this.updateTextField(currentValue);
    }

    public void setDecimalMode() {
        BigInteger currentValue = this.getValue();
        this.isHexMode = false;
        this.updateTextField(currentValue);
    }

    public void setAllowsHexPrefix(boolean allowsHexPrefix) {
        BigInteger currentValue = this.getValue();
        this.allowsHexPrefix = allowsHexPrefix;
        this.updateTextField(currentValue);
    }

    public String getText() {
        return this.textField.getText();
    }

    public boolean isHexMode() {
        return this.isHexMode;
    }

    public void setAllowNegativeValues(boolean b) {
        BigInteger currentValue = this.getValue();
        this.allowsNegative = b;
        if (!this.allowsNegative && currentValue != null && currentValue.compareTo(BigInteger.ZERO) < 0) {
            currentValue = null;
        }
        this.updateTextField(currentValue);
    }

    public BigInteger getMaxValue() {
        return this.maxValue;
    }

    public void setMaxValue(BigInteger maxValue) {
        if (maxValue != null && maxValue.compareTo(BigInteger.ZERO) < 0) {
            throw new IllegalArgumentException("Max value must be positive");
        }
        BigInteger currentValue = this.getValue();
        this.maxValue = maxValue;
        if (!this.passesMaxCheck(currentValue)) {
            this.setValue(maxValue);
        }
    }

    public JComponent getComponent() {
        return this.textField;
    }

    public void addActionListener(ActionListener listener) {
        this.textField.addActionListener(listener);
    }

    public void removeActionListener(ActionListener listener) {
        this.textField.removeActionListener(listener);
    }

    public void setEnabled(boolean enabled) {
        this.textField.setEnabled(enabled);
    }

    public void requestFocus() {
        this.textField.requestFocus();
    }

    public void selectAll() {
        this.textField.selectAll();
    }

    private String computeTextForValue(BigInteger value) {
        if (value == null) {
            return "";
        }
        if (this.isHexMode) {
            String text = value.toString(16);
            if (this.allowsHexPrefix) {
                if (text.startsWith("-")) {
                    return "-0x" + text.substring(1);
                }
                return "0x" + text;
            }
            return text;
        }
        return value.toString(10);
    }

    private BigInteger computeValueFromString(String text) {
        if (((String)text).isEmpty() || this.isValidPrefix((String)text)) {
            return null;
        }
        if (!this.isHexMode) {
            return new BigInteger((String)text, 10);
        }
        if (this.allowsHexPrefix) {
            if (((String)text).startsWith("0x")) {
                text = ((String)text).substring(2);
            } else if (((String)text).startsWith("-0x")) {
                text = "-" + ((String)text).substring(3);
            }
        }
        if (((String)text).isEmpty()) {
            return null;
        }
        return new BigInteger((String)text, 16);
    }

    private void valueChanged() {
        for (ChangeListener listener : this.listeners) {
            listener.stateChanged(new ChangeEvent(this));
        }
    }

    private void toggleMode() {
        if (this.isHexMode) {
            this.setDecimalMode();
        } else {
            this.setHexMode();
        }
    }

    private boolean passesMaxCheck(BigInteger value) {
        if (value == null) {
            return true;
        }
        if (this.maxValue == null) {
            return true;
        }
        return value.compareTo(this.maxValue) <= 0;
    }

    private void updateNumberMode(String text) {
        if (this.allowsHexPrefix) {
            this.isHexMode = text.contains("0x");
        }
    }

    private void updateTextField(BigInteger value) {
        String text = this.computeTextForValue(value);
        this.textField.setText(text);
    }

    private boolean isValidPrefix(String s) {
        switch (s) {
            case "0x": {
                return this.allowsHexPrefix;
            }
            case "-0x": {
                return this.allowsHexPrefix && this.allowsNegative;
            }
            case "-": {
                return this.allowsNegative;
            }
        }
        return false;
    }

    private class MyTextField
    extends JTextField {
        private Font hintFont;
        private int hintWidth;

        public MyTextField(int columns) {
            super(columns);
            this.hintFont = new Font("Monospaced", 0, 10);
            FontMetrics fontMetrics = this.getFontMetrics(this.hintFont);
            String mode = IntegerTextField.this.isHexMode ? "Hex" : "Dec";
            this.hintWidth = fontMetrics.stringWidth(mode);
            AbstractDocument document = (AbstractDocument)this.getDocument();
            document.setDocumentFilter(new HexDecimalDocumentFilter());
            this.addKeyListener(new KeyAdapter(){

                @Override
                public void keyPressed(KeyEvent e) {
                    if (e.getKeyCode() == 77 && DockingUtils.isControlModifier(e)) {
                        IntegerTextField.this.toggleMode();
                        MyTextField.this.repaint();
                    }
                }
            });
            ToolTipManager.sharedInstance().registerComponent(this);
        }

        @Override
        public String getToolTipText(MouseEvent event) {
            int hintStart = this.getBounds().width - this.hintWidth;
            if (event.getX() > hintStart) {
                String key = DockingUtils.CONTROL_KEY_NAME;
                return "Press '" + key + "-M' to toggle Hex or Decimal Mode";
            }
            return null;
        }

        @Override
        protected void paintComponent(Graphics g) {
            super.paintComponent(g);
            if (!IntegerTextField.this.showNumbericDecoration) {
                return;
            }
            Font savedFont = g.getFont();
            g.setFont(this.hintFont);
            g.setColor(Color.LIGHT_GRAY);
            Dimension size = this.getSize();
            Insets insets = this.getInsets();
            int x = size.width - insets.right - this.hintWidth;
            int y = size.height - insets.bottom - 1;
            String mode = IntegerTextField.this.isHexMode ? "Hex" : "Dec";
            GraphicsUtils.drawString((JComponent)this, g, mode, x, y);
            g.setFont(savedFont);
        }
    }

    private class HexDecimalDocumentFilter
    extends DocumentFilter {
        private HexDecimalDocumentFilter() {
        }

        @Override
        public void insertString(DocumentFilter.FilterBypass fb, int offset, String text, AttributeSet attr) throws BadLocationException {
            text = text.replace('X', 'x');
            StringBuilder builder = this.getText(fb);
            builder.insert(offset, text);
            if (this.isValid(builder)) {
                super.insertString(fb, offset, text, attr);
            }
        }

        @Override
        public void replace(DocumentFilter.FilterBypass fb, int offset, int length, String text, AttributeSet attrs) throws BadLocationException {
            text = text.replace('X', 'x');
            StringBuilder builder = this.getText(fb);
            builder.replace(offset, offset + length, text);
            if (this.isValid(builder)) {
                super.replace(fb, offset, length, text, attrs);
            }
        }

        @Override
        public void remove(DocumentFilter.FilterBypass fb, int offset, int length) throws BadLocationException {
            StringBuilder builder = this.getText(fb);
            builder.delete(offset, offset + length);
            if (this.isValid(builder)) {
                super.remove(fb, offset, length);
            }
        }

        private boolean isValid(StringBuilder builder) {
            String valueString = builder.toString();
            IntegerTextField.this.updateNumberMode(valueString);
            if (IntegerTextField.this.isValidPrefix(valueString)) {
                return true;
            }
            try {
                BigInteger value = IntegerTextField.this.computeValueFromString(valueString);
                if (!IntegerTextField.this.allowsNegative && value != null && value.signum() < 0) {
                    return false;
                }
                return IntegerTextField.this.passesMaxCheck(value);
            }
            catch (NumberFormatException e) {
                return false;
            }
        }

        private StringBuilder getText(DocumentFilter.FilterBypass fb) throws BadLocationException {
            StringBuilder builder = new StringBuilder();
            Document document = fb.getDocument();
            builder.append(document.getText(0, document.getLength()));
            return builder;
        }
    }
}

