//============================================================================
//
//  This file is part of GPSTk, the GPS Toolkit.
//
//  The GPSTk is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 3.0 of the License, or
//  any later version.
//
//  The GPSTk is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with GPSTk; if not, write to the Free Software Foundation,
//  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
//  
//  Copyright 2004, The University of Texas at Austin
//  Dagoberto Salazar - gAGE ( http://www.gage.es ). 2012
//
//============================================================================

//============================================================================
//
//This software developed by Applied Research Laboratories at the University of
//Texas at Austin, under contract to an agency or agencies within the U.S. 
//Department of Defense. The U.S. Government retains all rights to use,
//duplicate, distribute, disclose, or release this software. 
//
//Pursuant to DoD Directive 523024 
//
// DISTRIBUTION STATEMENT A: This software has been approved for public 
//                           release, distribution is unlimited.
//
//=============================================================================


/**
 * @file ExtractData.hpp
 * This is the base class to ease data extraction from a RinexObsData object.
 */

#ifndef GPSTK_EXTRACTDATA_HPP
#define GPSTK_EXTRACTDATA_HPP

#include "Exception.hpp"
#include "Rinex3ObsHeader.hpp"
#include "Rinex3ObsData.hpp"
#include "CheckPRData.hpp"
#include "Vector.hpp"


namespace gpstk
{

      /// @ingroup RinexObs 
      //@{


      /// This is the base class to ease data extraction from a RinexObsData object.
   class ExtractData
   {
   public:

         /// Return validity of data
      inline bool isValid(void) const
      { return valid; }


         /// Number of satellites with available data
      int numSV;


         /// Vector with the PRN of satellites with available data.
      Vector<SatID> availableSV;


         /// Vector holding the available data
      Vector<double> obsData;


         /// Default constructor
      ExtractData()
         : checkData(true), valid(false), minPRange(15000000.0),
           maxPRange(30000000.0)
      {};


         /** Pull out the selected observation type from a Rinex3ObsData object
          *
          * @param rinexData  The Rinex data set holding the observations
          * @param index      Index representing the observation type. It is
          *                   obtained from corresponding RINEX Obs Header
          *                   using method 'Rinex3ObsHeader::getObsIndex()'.
          *
          * @return
          *  Number of satellites with this kind of data available
          */
      virtual int getData( const Rinex3ObsData& rinexData, int index )
         throw(InvalidRequest);


         /** Pull out the selected observation type from a Rinex3ObsData object
          *
          * @param rinexData  The Rinex data set holding the observations
          * @param type       String representing the observation type.
          * @param hdr        RINEX Observation Header for current RINEX file.
          */
      virtual int getData( const Rinex3ObsData& rinexData,
                           std::string type,
                           const Rinex3ObsHeader& hdr )
         throw(InvalidRequest);


         /// Set this to true if you want to enable data checking within given
         /// boundaries (default for code measurements)
      bool checkData;


         /// Set the minimum pseudorange value allowed for data (in meters).
      virtual ExtractData& setMinPRange( double minPR )
      { minPRange = minPR; return (*this); };


         /// Get the minimum pseudorange value allowed for data (in meters).
      virtual double getMinPRange(void) const
      { return minPRange; };


         /// Set the maximum pseudorange value allowed for data (in meters).
      virtual ExtractData& setMaxPRange( double maxPR )
      { maxPRange = maxPR; return (*this); };


         /// Get the minimum pseudorange value allowed for data (in meters).
      virtual double getMaxPRange(void) const
      { return maxPRange; };


         /// Destructor
      virtual ~ExtractData() {};


   protected:


         /// True only if results are valid
      bool valid;


         /// Minimum pseudorange value allowed for input data (in meters).
      double minPRange;


         /// Maximum pseudorange value allowed for input data (in meters).
      double maxPRange;


   }; // End of class 'ExtractData'
   

      //@}
   
}  // End of namespace gpstk

#endif   // GPSTK_EXTRACTDATA_HPP
