/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import java.io.Closeable;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.ReadableInstant;
import org.logstash.DLQEntry;
import org.logstash.Event;
import org.logstash.FieldReference;
import org.logstash.FileLockFactory;
import org.logstash.Timestamp;
import org.logstash.common.io.RecordIOWriter;

public final class DeadLetterQueueWriter
implements Closeable {
    private static final Logger logger = LogManager.getLogger(DeadLetterQueueWriter.class);
    private static final long MAX_SEGMENT_SIZE_BYTES = 0xA00000L;
    static final String SEGMENT_FILE_PATTERN = "%d.log";
    static final String LOCK_FILE = ".lock";
    private static final FieldReference DEAD_LETTER_QUEUE_METADATA_KEY = FieldReference.from(String.format("%s[dead_letter_queue]", "[@metadata]"));
    private final long maxSegmentSize;
    private final long maxQueueSize;
    private LongAdder currentQueueSize;
    private final Path queuePath;
    private final FileLock lock;
    private volatile RecordIOWriter currentWriter;
    private int currentSegmentIndex;
    private Timestamp lastEntryTimestamp;
    private final AtomicBoolean open = new AtomicBoolean(true);

    public DeadLetterQueueWriter(Path queuePath, long maxSegmentSize, long maxQueueSize) throws IOException {
        this.lock = FileLockFactory.obtainLock(queuePath, LOCK_FILE);
        this.queuePath = queuePath;
        this.maxSegmentSize = maxSegmentSize;
        this.maxQueueSize = maxQueueSize;
        this.currentQueueSize = new LongAdder();
        this.currentQueueSize.add(this.getStartupQueueSize());
        this.currentSegmentIndex = DeadLetterQueueWriter.getSegmentPaths(queuePath).map(s -> s.getFileName().toString().split("\\.")[0]).mapToInt(Integer::parseInt).max().orElse(0);
        this.nextWriter();
        this.lastEntryTimestamp = Timestamp.now();
    }

    public DeadLetterQueueWriter(String queuePath) throws IOException {
        this(Paths.get(queuePath, new String[0]), 0xA00000L, Long.MAX_VALUE);
    }

    private long getStartupQueueSize() throws IOException {
        return DeadLetterQueueWriter.getSegmentPaths(this.queuePath).mapToLong(p -> {
            try {
                return Files.size(p);
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }).sum();
    }

    private void nextWriter() throws IOException {
        this.currentWriter = new RecordIOWriter(this.queuePath.resolve(String.format(SEGMENT_FILE_PATTERN, ++this.currentSegmentIndex)));
        this.currentQueueSize.increment();
    }

    static Stream<Path> getSegmentPaths(Path path) throws IOException {
        try (Stream<Path> files = Files.list(path);){
            Stream<Path> stream = files.filter(p -> p.toString().endsWith(".log")).collect(Collectors.toList()).stream();
            return stream;
        }
    }

    public synchronized void writeEntry(DLQEntry entry) throws IOException {
        this.innerWriteEntry(entry);
    }

    public synchronized void writeEntry(Event event, String pluginName, String pluginId, String reason) throws IOException {
        Timestamp entryTimestamp = Timestamp.now();
        if (entryTimestamp.getTime().isBefore((ReadableInstant)this.lastEntryTimestamp.getTime())) {
            entryTimestamp = this.lastEntryTimestamp;
        }
        DLQEntry entry = new DLQEntry(event, pluginName, pluginId, reason);
        this.innerWriteEntry(entry);
        this.lastEntryTimestamp = entryTimestamp;
    }

    private void innerWriteEntry(DLQEntry entry) throws IOException {
        Event event = entry.getEvent();
        if (DeadLetterQueueWriter.alreadyProcessed(event)) {
            logger.warn("Event previously submitted to dead letter queue. Skipping...");
            return;
        }
        byte[] record = entry.serialize();
        int eventPayloadSize = 13 + record.length;
        if (this.currentQueueSize.longValue() + (long)eventPayloadSize > this.maxQueueSize) {
            logger.error("cannot write event to DLQ: reached maxQueueSize of " + this.maxQueueSize);
            return;
        }
        if (this.currentWriter.getPosition() + (long)eventPayloadSize > this.maxSegmentSize) {
            this.currentWriter.close();
            this.nextWriter();
        }
        this.currentQueueSize.add(this.currentWriter.writeEvent(record));
    }

    private static boolean alreadyProcessed(Event event) {
        return event.includes(DEAD_LETTER_QUEUE_METADATA_KEY);
    }

    @Override
    public void close() {
        if (this.open.compareAndSet(true, false)) {
            if (this.currentWriter != null) {
                try {
                    this.currentWriter.close();
                }
                catch (Exception e) {
                    logger.debug("Unable to close dlq writer", (Throwable)e);
                }
            }
            this.releaseLock();
        }
    }

    private void releaseLock() {
        try {
            FileLockFactory.releaseLock(this.lock);
        }
        catch (IOException e) {
            logger.debug("Unable to release lock", (Throwable)e);
        }
        try {
            Files.deleteIfExists(this.queuePath.resolve(LOCK_FILE));
        }
        catch (IOException e) {
            logger.debug("Unable to delete lock file", (Throwable)e);
        }
    }

    public boolean isOpen() {
        return this.open.get();
    }

    public Path getPath() {
        return this.queuePath;
    }

    public long getCurrentQueueSize() {
        return this.currentQueueSize.longValue();
    }
}

