/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.persistent;

import java.io.Closeable;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.AbstractAsyncTask;
import org.elasticsearch.persistent.PersistentTaskParams;
import org.elasticsearch.persistent.PersistentTaskState;
import org.elasticsearch.persistent.PersistentTasksCustomMetaData;
import org.elasticsearch.persistent.PersistentTasksExecutor;
import org.elasticsearch.persistent.PersistentTasksExecutorRegistry;
import org.elasticsearch.persistent.decider.AssignmentDecision;
import org.elasticsearch.persistent.decider.EnableAssignmentDecider;
import org.elasticsearch.threadpool.ThreadPool;

public class PersistentTasksClusterService
implements ClusterStateListener,
Closeable {
    public static final Setting<TimeValue> CLUSTER_TASKS_ALLOCATION_RECHECK_INTERVAL_SETTING = Setting.timeSetting("cluster.persistent_tasks.allocation.recheck_interval", TimeValue.timeValueSeconds((long)30L), TimeValue.timeValueSeconds((long)10L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    private static final Logger logger = LogManager.getLogger(PersistentTasksClusterService.class);
    private final ClusterService clusterService;
    private final PersistentTasksExecutorRegistry registry;
    private final EnableAssignmentDecider decider;
    private final ThreadPool threadPool;
    private final PeriodicRechecker periodicRechecker;

    public PersistentTasksClusterService(Settings settings, PersistentTasksExecutorRegistry registry, ClusterService clusterService, ThreadPool threadPool) {
        this.clusterService = clusterService;
        this.registry = registry;
        this.decider = new EnableAssignmentDecider(settings, clusterService.getClusterSettings());
        this.threadPool = threadPool;
        this.periodicRechecker = new PeriodicRechecker(CLUSTER_TASKS_ALLOCATION_RECHECK_INTERVAL_SETTING.get(settings));
        clusterService.addListener(this);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(CLUSTER_TASKS_ALLOCATION_RECHECK_INTERVAL_SETTING, this::setRecheckInterval);
    }

    public void setRecheckInterval(TimeValue recheckInterval) {
        this.periodicRechecker.setInterval(recheckInterval);
    }

    @Override
    public void close() {
        this.periodicRechecker.close();
    }

    public <Params extends PersistentTaskParams> void createPersistentTask(final String taskId, final String taskName, final Params taskParams, final ActionListener<PersistentTasksCustomMetaData.PersistentTask<?>> listener) {
        this.clusterService.submitStateUpdateTask("create persistent task", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                PersistentTasksCustomMetaData.Builder builder = PersistentTasksClusterService.builder(currentState);
                if (builder.hasTask(taskId)) {
                    throw new ResourceAlreadyExistsException("task with id {" + taskId + "} already exist", new Object[0]);
                }
                PersistentTasksExecutor<PersistentTaskParams> taskExecutor = PersistentTasksClusterService.this.registry.getPersistentTaskExecutorSafe(taskName);
                taskExecutor.validate(taskParams, currentState);
                PersistentTasksCustomMetaData.Assignment assignment = PersistentTasksClusterService.this.createAssignment(taskName, taskParams, currentState);
                return PersistentTasksClusterService.update(currentState, builder.addTask(taskId, taskName, taskParams, assignment));
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)newState.getMetaData().custom("persistent_tasks");
                if (tasks != null) {
                    PersistentTasksCustomMetaData.PersistentTask<?> task = tasks.getTask(taskId);
                    listener.onResponse(task);
                    if (task != null && !task.isAssigned() && !PersistentTasksClusterService.this.periodicRechecker.isScheduled()) {
                        PersistentTasksClusterService.this.periodicRechecker.rescheduleIfNecessary();
                    }
                } else {
                    listener.onResponse(null);
                }
            }
        });
    }

    public void completePersistentTask(final String id, final long allocationId, Exception failure, final ActionListener<PersistentTasksCustomMetaData.PersistentTask<?>> listener) {
        String source;
        if (failure != null) {
            logger.warn("persistent task " + id + " failed", (Throwable)failure);
            source = "finish persistent task (failed)";
        } else {
            source = "finish persistent task (success)";
        }
        this.clusterService.submitStateUpdateTask(source, new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                PersistentTasksCustomMetaData.Builder tasksInProgress = PersistentTasksClusterService.builder(currentState);
                if (tasksInProgress.hasTask(id, allocationId)) {
                    tasksInProgress.removeTask(id);
                    return PersistentTasksClusterService.update(currentState, tasksInProgress);
                }
                if (tasksInProgress.hasTask(id)) {
                    logger.warn("The task [{}] with id [{}] was found but it has a different allocation id [{}], status is not updated", (Object)PersistentTasksCustomMetaData.getTaskWithId(currentState, id).getTaskName(), (Object)id, (Object)allocationId);
                } else {
                    logger.warn("The task [{}] wasn't found, status is not updated", (Object)id);
                }
                throw new ResourceNotFoundException("the task with id [" + id + "] and allocation id [" + allocationId + "] not found", new Object[0]);
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                listener.onResponse(PersistentTasksCustomMetaData.getTaskWithId(oldState, id));
            }
        });
    }

    public void removePersistentTask(final String id, final ActionListener<PersistentTasksCustomMetaData.PersistentTask<?>> listener) {
        this.clusterService.submitStateUpdateTask("remove persistent task", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                PersistentTasksCustomMetaData.Builder tasksInProgress = PersistentTasksClusterService.builder(currentState);
                if (tasksInProgress.hasTask(id)) {
                    return PersistentTasksClusterService.update(currentState, tasksInProgress.removeTask(id));
                }
                throw new ResourceNotFoundException("the task with id {} doesn't exist", id);
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                listener.onResponse(PersistentTasksCustomMetaData.getTaskWithId(oldState, id));
            }
        });
    }

    public void updatePersistentTaskState(final String taskId, final long taskAllocationId, final PersistentTaskState taskState, final ActionListener<PersistentTasksCustomMetaData.PersistentTask<?>> listener) {
        this.clusterService.submitStateUpdateTask("update task state", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                PersistentTasksCustomMetaData.Builder tasksInProgress = PersistentTasksClusterService.builder(currentState);
                if (tasksInProgress.hasTask(taskId, taskAllocationId)) {
                    return PersistentTasksClusterService.update(currentState, tasksInProgress.updateTaskState(taskId, taskState));
                }
                if (tasksInProgress.hasTask(taskId)) {
                    logger.warn("trying to update state on task {} with unexpected allocation id {}", (Object)taskId, (Object)taskAllocationId);
                } else {
                    logger.warn("trying to update state on non-existing task {}", (Object)taskId);
                }
                throw new ResourceNotFoundException("the task with id {} and allocation id {} doesn't exist", taskId, taskAllocationId);
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                listener.onResponse(PersistentTasksCustomMetaData.getTaskWithId(newState, taskId));
            }
        });
    }

    private <Params extends PersistentTaskParams> PersistentTasksCustomMetaData.Assignment createAssignment(String taskName, Params taskParams, ClusterState currentState) {
        PersistentTasksExecutor<Params> persistentTasksExecutor = this.registry.getPersistentTaskExecutorSafe(taskName);
        AssignmentDecision decision = this.decider.canAssign();
        if (decision.getType() == AssignmentDecision.Type.NO) {
            return new PersistentTasksCustomMetaData.Assignment(null, "persistent task [" + taskName + "] cannot be assigned [" + decision.getReason() + "]");
        }
        return persistentTasksExecutor.getAssignment(taskParams, currentState);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (event.localNodeMaster() && this.shouldReassignPersistentTasks(event)) {
            this.periodicRechecker.cancel();
            logger.trace("checking task reassignment for cluster state {}", (Object)event.state().getVersion());
            this.reassignPersistentTasks();
        }
    }

    private void reassignPersistentTasks() {
        this.clusterService.submitStateUpdateTask("reassign persistent tasks", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                return PersistentTasksClusterService.this.reassignTasks(currentState);
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.warn("failed to reassign persistent tasks", (Throwable)e);
                PersistentTasksClusterService.this.periodicRechecker.rescheduleIfNecessary();
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                if (PersistentTasksClusterService.this.isAnyTaskUnassigned((PersistentTasksCustomMetaData)newState.getMetaData().custom("persistent_tasks"))) {
                    PersistentTasksClusterService.this.periodicRechecker.rescheduleIfNecessary();
                }
            }
        });
    }

    boolean shouldReassignPersistentTasks(ClusterChangedEvent event) {
        boolean masterChanged;
        PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)event.state().getMetaData().custom("persistent_tasks");
        if (tasks == null) {
            return false;
        }
        boolean bl = masterChanged = !event.previousState().nodes().isLocalNodeElectedMaster();
        if (PersistentTasksClusterService.persistentTasksChanged(event) || event.nodesChanged() || event.routingTableChanged() || event.metaDataChanged() || masterChanged) {
            for (PersistentTasksCustomMetaData.PersistentTask<?> task : tasks.tasks()) {
                PersistentTasksCustomMetaData.Assignment assignment;
                if (!PersistentTasksClusterService.needsReassignment(task.getAssignment(), event.state().nodes()) || Objects.equals(assignment = this.createAssignment(task.getTaskName(), (PersistentTaskParams)task.getParams(), event.state()), task.getAssignment())) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isAnyTaskUnassigned(PersistentTasksCustomMetaData tasks) {
        return tasks != null && tasks.tasks().stream().anyMatch(task -> !task.getAssignment().isAssigned());
    }

    ClusterState reassignTasks(ClusterState currentState) {
        ClusterState clusterState = currentState;
        PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)currentState.getMetaData().custom("persistent_tasks");
        if (tasks != null) {
            logger.trace("reassigning {} persistent tasks", (Object)tasks.tasks().size());
            DiscoveryNodes nodes = currentState.nodes();
            for (PersistentTasksCustomMetaData.PersistentTask<?> task : tasks.tasks()) {
                if (PersistentTasksClusterService.needsReassignment(task.getAssignment(), nodes)) {
                    PersistentTasksCustomMetaData.Assignment assignment = this.createAssignment(task.getTaskName(), (PersistentTaskParams)task.getParams(), clusterState);
                    if (!Objects.equals(assignment, task.getAssignment())) {
                        logger.trace("reassigning task {} from node {} to node {}", (Object)task.getId(), (Object)task.getAssignment().getExecutorNode(), (Object)assignment.getExecutorNode());
                        clusterState = PersistentTasksClusterService.update(clusterState, PersistentTasksClusterService.builder(clusterState).reassignTask(task.getId(), assignment));
                        continue;
                    }
                    logger.trace("ignoring task {} because assignment is the same {}", (Object)task.getId(), (Object)assignment);
                    continue;
                }
                logger.trace("ignoring task {} because it is still running", (Object)task.getId());
            }
        }
        return clusterState;
    }

    static boolean persistentTasksChanged(ClusterChangedEvent event) {
        String type = "persistent_tasks";
        return !Objects.equals(event.state().metaData().custom(type), event.previousState().metaData().custom(type));
    }

    public static boolean needsReassignment(PersistentTasksCustomMetaData.Assignment assignment, DiscoveryNodes nodes) {
        return !assignment.isAssigned() || !nodes.nodeExists(assignment.getExecutorNode());
    }

    private static PersistentTasksCustomMetaData.Builder builder(ClusterState currentState) {
        return PersistentTasksCustomMetaData.builder((PersistentTasksCustomMetaData)currentState.getMetaData().custom("persistent_tasks"));
    }

    private static ClusterState update(ClusterState currentState, PersistentTasksCustomMetaData.Builder tasksInProgress) {
        if (tasksInProgress.isChanged()) {
            return ClusterState.builder(currentState).metaData(MetaData.builder(currentState.metaData()).putCustom("persistent_tasks", tasksInProgress.build())).build();
        }
        return currentState;
    }

    private class PeriodicRechecker
    extends AbstractAsyncTask {
        PeriodicRechecker(TimeValue recheckInterval) {
            super(logger, PersistentTasksClusterService.this.threadPool, recheckInterval, false);
        }

        @Override
        protected boolean mustReschedule() {
            return true;
        }

        @Override
        public void runInternal() {
            if (PersistentTasksClusterService.this.clusterService.localNode().isMasterNode()) {
                ClusterState state = PersistentTasksClusterService.this.clusterService.state();
                logger.trace("periodic persistent task assignment check running for cluster state {}", (Object)state.getVersion());
                if (PersistentTasksClusterService.this.isAnyTaskUnassigned((PersistentTasksCustomMetaData)state.getMetaData().custom("persistent_tasks"))) {
                    PersistentTasksClusterService.this.reassignPersistentTasks();
                }
            }
        }

        public String toString() {
            return "persistent_task_recheck";
        }
    }
}

